/*!
 * @file        main.c
 *
 * @brief       Main program body
 *
 * @version     V1.0.2
 *
 * @date        2022-06-23
 *
 * @attention
 *
 *  Copyright (C) 2021-2022 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be usefull and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes */
#include "main.h"
#include "Board.h"
#include "apm32f4xx_rcm.h"
#include "apm32f4xx_rng.h"
#include "apm32f4xx_usart.h"
#include "stdio.h"

/** @addtogroup Examples
  @{
  */

/** @addtogroup RNG_MultiRNG
  @{
  */

/** @defgroup RNG_MultiRNG_Macros Macros
  @{
*/

/* printf using USART1  */
#define DEBUG_USART  USART1

/**@} end of group RNG_MultiRNG_Macros*/

/** @defgroup RNG_MultiRNG_Functions Functions
  @{
  */

/* RNG initialization */
void RNG_Init(void);

/* Delay */
void Delay(uint32_t count);

/*!
 * @brief       Main program
 *
 * @param       None
 *
 * @retval      None
 */
int main(void)
{
    USART_Config_T usartConfigStruct;
    uint32_t uRandomNumber = 0;

    /* RNG initialization */
    RNG_Init();

    usartConfigStruct.baudRate = 115200;
    usartConfigStruct.hardwareFlow = USART_HARDWARE_FLOW_NONE;
    usartConfigStruct.mode = USART_MODE_TX;
    usartConfigStruct.parity = USART_PARITY_NONE;
    usartConfigStruct.stopBits = USART_STOP_BIT_1;
    usartConfigStruct.wordLength = USART_WORD_LEN_8B;
    APM_MINI_COMInit(COM1, &usartConfigStruct);

    while (1)
    {
        uRandomNumber = RNG_ReadRandomNumber();
        printf("RandomNumber = 0x%08X\r\n",uRandomNumber);

        Delay(0x7FFFFF);
    }
}

/*!
 * @brief       RNG initialization
 *
 * @param       None
 *
 * @retval      None
 */
void RNG_Init(void)
{
    RCM_EnableAHB2PeriphClock(RCM_AHB2_PERIPH_RNG);
    RNG_Enable();
}

/*!
 * @brief       Delay
 *
 * @param       count:  delay count
 *
 * @retval      None
 */
void Delay(uint32_t count)
{
    volatile uint32_t delay = count;
    while(delay--);
}

/*!
 * @brief       Redirect C Library function printf to serial port.
 *              After Redirection, you can use printf function.
 *
 * @param       ch:  The characters that need to be send.
 *
 * @param       *f:  pointer to a FILE that can recording all information
 *              needed to control a stream
 *
 * @retval      The characters that need to be send.
 */
int fputc(int ch, FILE *f)
{
    /* send a byte of data to the serial port */
    USART_TxData(DEBUG_USART,(uint8_t)ch);

    /* wait for the data to be send  */
    while (USART_ReadStatusFlag(DEBUG_USART, USART_FLAG_TXBE) == RESET);

    return (ch);
}

/**@} end of group RNG_MultiRNG_Functions */
/**@} end of group RNG_MultiRNG */
/**@} end of group Examples */
